#include <linux/delay.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/input/mt.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/workqueue.h>
#include <linux/earlysuspend.h>
#include <linux/firmware.h>
#include <linux/regulator/consumer.h>
#include <linux/device.h>

#include <linux/ft5x06_i2c_defs.h>
#include "ft5x06_ts_i2c.h"
#include <asm/atomic.h>

#include <linux/crestron.h>

#include "crestron/crestSoftKeys.h"

/* Move these all into a common location. */
extern int uiTouchIntfInit(struct crestSoftKeys_ctrl_t* ctrl);

#define ETML_NAME      "etml_ts"


#define ETML_QUERY_STR_SIZE (16)

struct etml {
    struct i2c_client *client;
    struct crestSoftKeys_ctrl_t ctrl;

    struct work_struct work;
    struct timer_list timer;
    int do_reset;
    int reset_enable;
    int reset_time;

    char phys[32];
    struct ft5x06_platform_data *platform_data;
    struct timeval last_rd_time;

    u8 tch_filter_rate;
    u8 filter_count;

    char fwver[ETML_QUERY_STR_SIZE];
    char fwname[ETML_QUERY_STR_SIZE];
    /* Ensures that only one function can specify the Device Mode at a time. */
    struct mutex device_mode_mutex;
    struct early_suspend early_suspend;
    struct regulator *vtp;
};

/*************************************************************************
 * GPIO Helper Functions
 ************************************************************************/

static void etml_reset_panel_via_gpio(int reset_gpio)
{
    gpio_direction_output(reset_gpio, 1);
    msleep(5);
    gpio_direction_output(reset_gpio, 0);
    msleep(8);
    gpio_direction_output(reset_gpio, 1);
    msleep(50);
}

/**
 * 1050 defines.
 */
#define MAX_NUM_PACKETS_READ_1050 5

#define ETML_PACKET_SIZE (10)

#define GET_PACKET_ID_1050(buf)  ((buf)[0])
#define GET_PACKET_VALID_1050(buf) ((buf)[1] >> 7)
#define GET_EVENT_1050(buf) ((buf)[1] & 0x01)
#define GET_TOUCH_ID_1050(buf) ((int) (((buf)[1] & 0x7C) >> 2))

/**
 * Kind of hairy, but this is used to bring the X coordinate into the proper
 * range when GET_XCOORD_1050 is used.
 */
#define CALIBRATE_VAL_X 108
#define TSW1050_CALIBRATE_X(x) ((u16)(((x)*CALIBRATE_VAL_X)/100)) //(need to accomodate for Vbutton area)

// Divide coordinates by 16 (32765 original), manipulate x to accomodate for VButton area.
#define GET_XCOORD_1050(buf) ( TSW1050_CALIBRATE_X(((buf)[3] << 8 | (buf)[2]) >> 4) )
#define GET_YCOORD_1050(buf) ((buf[5] << 8 | buf[4]) >> 4)
#define GET_ZCOORD_1050(buf) ((buf[7] << 8 | buf[6]) >> 4)

// Minimum delay between packet reads for the 1050, in us.
#define PACKET_READ_DELAY_1050 (60)


// Diagnostic packet defines
#define DIAG_QUERY_READ_DELAY_MS (80)
#define DIAG_QUERY_READ_RETRIES  ( 2)

#define ETML_DEBUG(X)
//#define ETML_DEBUG(X) X


static irqreturn_t etml_irq(int irq, void *handle);
static void etml_xy_worker(struct work_struct *work);

static struct workqueue_struct *etml_ts_wq;

static const struct i2c_device_id etml_ts_id[] = {
    { ETML_NAME, 0 },  { }
};
MODULE_DEVICE_TABLE(i2c, etml_ts_id);

#define ETML_SCAN_DELAY               20

DECLARE_COMPLETION(name_query_comp);
DECLARE_COMPLETION(version_query_comp);

static void etml_sendNameQueryCommand(struct etml *ts)
{
	u8 queryCmd[ETML_PACKET_SIZE] = {0x03,0x03,0x0a,0x01,'E',0,0,0,0,0};
	int retval = i2c_master_send(ts->client, (u8*)&queryCmd, sizeof(queryCmd));
	if(retval != ETML_PACKET_SIZE) {
		dev_err(&ts->client->dev, "i2c error %i: Failed to send Name command\n", retval);
	}
}

static void etml_sendVersionQueryCommand(struct etml *ts)
{
	u8 queryCmd[ETML_PACKET_SIZE] = {0x03,0x03,0x0a,0x01,'D',0,0,0,0,0};
	int retval = i2c_master_send(ts->client, (u8*)&queryCmd, sizeof(queryCmd));
	if(retval != ETML_PACKET_SIZE) {
		dev_err(&ts->client->dev, "i2c error %i: Failed to send Version command\n", retval);
	}
}

/* sysfs */
static ssize_t etml_tpname_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct etml *ts = (struct etml *)i2c_get_clientdata(client);
    ssize_t numBytes = 0;

    /* Assume we haven't run the command if the first byte is still 0. */
    int has_response = 0;
    mutex_lock(&ts->device_mode_mutex);
    has_response = ts->fwname[0];
	mutex_unlock(&ts->device_mode_mutex);

	if(!has_response) {
	    etml_sendNameQueryCommand(ts);
	    wait_for_completion(&name_query_comp);
	}

    mutex_lock(&ts->device_mode_mutex);
    numBytes = scnprintf(buf, PAGE_SIZE, "%s\n", ts->fwname);
	mutex_unlock(&ts->device_mode_mutex);
    return numBytes;
}

static ssize_t etml_tpfwver_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct etml *ts = (struct etml *)i2c_get_clientdata(client);
    ssize_t numBytes = 0;

    /* Assume we haven't run the command if the first byte is still 0. */
    int has_response = 0;
    mutex_lock(&ts->device_mode_mutex);
    has_response = ts->fwver[0];
	mutex_unlock(&ts->device_mode_mutex);

	if(!has_response) {
		etml_sendVersionQueryCommand(ts);
		wait_for_completion(&version_query_comp);
	}

    mutex_lock(&ts->device_mode_mutex);
    numBytes = scnprintf(buf, PAGE_SIZE, "%s\n", ts->fwver);
    mutex_unlock(&ts->device_mode_mutex);
    return numBytes;
}

static void etml_reset_timer(struct etml *ts)
{
	if(ts->reset_enable && ts->reset_time)
		mod_timer(&ts->timer, jiffies + msecs_to_jiffies(ts->reset_time*1000));
	else
		del_timer(&ts->timer);
}

static void etml_timer_reset_controller(unsigned long data)
{
	struct etml *ts = (struct etml *)data;

	ts->do_reset = 1;
    queue_work(etml_ts_wq, &ts->work);

    etml_reset_timer(ts);

    return;
}
/**
 * This only exists to work around a firmware bug in an early revision of the
 * touch controller that causes it to lockup around 6 days 8 hours. Note that
 * this only works if the reset gpio is connected.
 */
static ssize_t etml_reset_controller(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct etml *ts = (struct etml *)i2c_get_clientdata(client);

    ts->do_reset = 1;
	queue_work(etml_ts_wq, &ts->work);
    return size;
}

static ssize_t etml_resetenable_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = container_of(dev, struct i2c_client, dev);
	struct etml *ts = (struct etml *)i2c_get_clientdata(client);
	ssize_t numBytes = 0;

	mutex_lock(&ts->device_mode_mutex);
	numBytes = scnprintf(buf, PAGE_SIZE, "%d\n", ts->reset_enable);
	mutex_unlock(&ts->device_mode_mutex);
	return numBytes;
}

static ssize_t etml_resetenable_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct etml *ts = (struct etml *)i2c_get_clientdata(client);

    sscanf(buf,"%d",&ts->reset_enable);
    etml_reset_timer(ts);
    return size;
}

static ssize_t etml_resettime_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = container_of(dev, struct i2c_client, dev);
	struct etml *ts = (struct etml *)i2c_get_clientdata(client);
	ssize_t numBytes = 0;

	mutex_lock(&ts->device_mode_mutex);
	numBytes = scnprintf(buf, PAGE_SIZE, "%d\n", ts->reset_time);
	mutex_unlock(&ts->device_mode_mutex);
	return numBytes;
}

static ssize_t etml_resettime_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct etml *ts = (struct etml *)i2c_get_clientdata(client);

    sscanf(buf,"%d",&ts->reset_time);
    etml_reset_timer(ts);
    return size;
}


static ssize_t etml_deep_sleep(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
	u8 sleepCmd[ETML_PACKET_SIZE] = {0x03,0x05,0x0a,0x03,0x36,0x3f,0x02,0,0,0};
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);

    int retval = i2c_master_send(client, (u8*)&sleepCmd, sizeof(sleepCmd));
	if(retval != ETML_PACKET_SIZE) {
		dev_err(&client->dev, "i2c error %i: Failed to send deep sleep command\n", retval);
	}
	return size;
}

static DEVICE_ATTR(tpname,  		S_IRUGO,			etml_tpname_show,		NULL);
static DEVICE_ATTR(tpfwver,			S_IRUGO,			etml_tpfwver_show,		NULL);
static DEVICE_ATTR(reset,			S_IWUGO,			NULL,					etml_reset_controller);
static DEVICE_ATTR(reset_enable,	S_IRUGO | S_IWUGO,	etml_resetenable_show,	etml_resetenable_store);
static DEVICE_ATTR(reset_time,		S_IRUGO | S_IWUGO,	etml_resettime_show,	etml_resettime_store);
static DEVICE_ATTR(deep_sleep,		S_IWUGO,			NULL,					etml_deep_sleep);
static struct attribute *etml_attributes[] = {
    &dev_attr_tpfwver.attr,
    &dev_attr_tpname.attr,
    &dev_attr_reset.attr,
    &dev_attr_reset_enable.attr,
    &dev_attr_reset_time.attr,
    &dev_attr_deep_sleep.attr,
    NULL
};
static struct attribute_group etml_attribute_group = {
    .attrs = etml_attributes
};

static ssize_t etml_touch_show_event(struct device *dev,struct device_attribute *attr, char *buf)
{
	return 0;
}

static DEVICE_ATTR(event, 				S_IRUGO,			etml_touch_show_event,				NULL);

static struct attribute *etml_touch_attributes[] = {
	&dev_attr_event.attr,
	NULL
};

static struct attribute_group etml_touch_attribute_group = {
    .attrs = etml_touch_attributes
};

/**
 * Wraps the i2c read in order to timestamp and dump reads for debugging.
 */
inline int etml_i2c_read(struct etml *ts, u8 buf[ETML_PACKET_SIZE])
{
	u32 bytesRead = 0;
	ETML_DEBUG( \
		char prefixBuf[64];
		struct timeval now; \
		__kernel_suseconds_t tdiff; \
		do_gettimeofday(&now); \
		tdiff = now.tv_usec - ts->last_rd_time.tv_usec; \
		ts->last_rd_time = now; \
	)

	bytesRead = i2c_master_recv(ts->client, (u8 *)buf, ETML_PACKET_SIZE);

	ETML_DEBUG( \
		snprintf(prefixBuf, 64, "%8li packetId=%i ", tdiff, buf[0]); \
		print_hex_dump(KERN_INFO, &prefixBuf[0], DUMP_PREFIX_ADDRESS, \
			16, 1, &buf[0], ETML_PACKET_SIZE, true); \
	)
	return bytesRead;
}

/**
 * Intended to handle all diagnostic packets (those with packet id 3).
 */
void etml_processDiagnosticPacket(struct etml *ts, u8 firstHalf[ETML_PACKET_SIZE])
{
	int strSize;
	int bytesRead = 0;
	u8 secondHalf[ETML_PACKET_SIZE];
	char *destBuf = 0;
	int retries = 0;
	struct completion *comp;
	ETML_DEBUG( struct timeval now; __kernel_suseconds_t tdiff;)
	const char *desc = "unknown";

	/* Handle wake packet and unknown packet types */
	if(!memcmp(&firstHalf[1], "\x05\x0A", 2)) {
		dev_info(&ts->client->dev, "Waking from idle/sleep.\n");
		return;
	}
	else if(memcmp(&firstHalf[1], "\x08\x0A", 2)) {
		// If this isn't the only other valid diagnostic packet...
		dev_err(&ts->client->dev, "Unknown diagnostic packet.\n");
		return;
	}
	/* Determine if this is a name or version response */
	if(firstHalf[4] == 'D') {
		destBuf = ts->fwver;
		desc = "version";
		comp = &version_query_comp;
	}
	else if(firstHalf[4] == 'E') {
		destBuf = ts->fwname;
		desc = "name";
		comp = &name_query_comp;
	}
	else {
		dev_err(&ts->client->dev, "Unknown diagnostic packet.\n");
		return;
	}

	/**
	 * Query responses come in two packets, but no other packets should come
	 * between them, wait until we can read the 2nd half.
	 */
	while(retries < DIAG_QUERY_READ_RETRIES) {
		// Wait for the interrupt line to go low again
		msleep(DIAG_QUERY_READ_DELAY_MS);
		if(!ts->platform_data->hw_status()) {
			bytesRead = etml_i2c_read(ts, &secondHalf[0]);
			if(bytesRead == ETML_PACKET_SIZE) {
				break;
			}
		}
		++retries;
	}
	if(retries == DIAG_QUERY_READ_RETRIES) {
		dev_err(&ts->client->dev, "Failed to query device.\n");
		return;
	}
	/* Not using the byte count for now. Should be used to only place the
	 * proper string contents into the output strings. */
	/* TODO: More error checking on secondHalf */
	strSize = firstHalf[3];
	mutex_lock(&ts->device_mode_mutex);
	memcpy(destBuf, &firstHalf[5], ETML_PACKET_SIZE-5);
	strncpy(&destBuf[ETML_PACKET_SIZE-5], &secondHalf[2], (ETML_PACKET_SIZE-2));
	dev_info(&ts->client->dev, "Touch controller %s: %s\n", desc, destBuf);
	mutex_unlock(&ts->device_mode_mutex);
	complete(comp);
}

/**
 * Reads the actual hardware and passes the data on to be processed.
 */
void etml_Read_TouchData_1050(struct etml *ts)
{
	u32 bytesRead=0;
	u8 ReadBuffer[ETML_PACKET_SIZE];
	int touchId = 0;// The touch ID of a particular touch-point
	u8 packetId = 0;// The packet ID of I2C read packet
	u8 isValid = 0;	// The packet ID of I2C read packet

	struct crest_xydata_t touch;
    u32 idProcessed = 0;

    ETML_DEBUG( \
		struct timeval now; \
		__kernel_suseconds_t tdiff;
		do_gettimeofday(&now); \
    )

    int fifoEvents = 0;
	/* Only read if the interrupt line is being held low */
    while(!ts->platform_data->hw_status())
	{
		// Clear local data-transfer buffer.
		memset(&ReadBuffer, 0, sizeof(ReadBuffer));
		// Per spec, needs at least 60 microsecond delay between reads
		//However, in practice it does not seem to need it. Processing consumes
		//enough time except when the diagnostic packets are involved.
		bytesRead = etml_i2c_read(ts, &ReadBuffer[0]);
		if(bytesRead < ETML_PACKET_SIZE) {
			printk(KERN_ERR "1050: Only read %i bytes.\n", bytesRead);
			continue;
		}

		packetId= GET_PACKET_ID_1050(ReadBuffer); // 0x04 cmd or 0x03 diag
		isValid = GET_PACKET_VALID_1050(ReadBuffer);// is read valid

		if (isValid && packetId == 0x04)
		{
			touchId = GET_TOUCH_ID_1050(ReadBuffer);  // id=0 or id=1  max 2 touches
			touch.state = GET_EVENT_1050(ReadBuffer);	  // B0=1: Touch Down; B0=0: Lift Off

			// We only report touch data when we've gotten a packet with the same
			//touchId a second time *since* we last sent data up. Presumably, this
			//means we will report data for all simultaneous touches.
			idProcessed |= (1 << touchId);

			//Parse X Y Z coords; nothing done with Z yet.
			touch.pos[0] = GET_XCOORD_1050(ReadBuffer);
			touch.pos[1] = GET_YCOORD_1050(ReadBuffer);

			/* This doesn't force sending the event up, only "significant" events
             *will be sent as a result of this line.*/
			crestSoftKeys_processTouchData(&ts->ctrl, touchId, touch, false);

			//TODO: Remove fifoEvents
			++fifoEvents;
		}
		else if(packetId == 0x03)
		{
			etml_processDiagnosticPacket(ts, ReadBuffer);
		}
		else
		{
			printk(KERN_INFO "%s: Unrecognized touch packet ID = 0x%x.\n", __FUNCTION__, packetId);
			msleep(PACKET_READ_DELAY_1050);
		}
	}
    /* This loop captures any continuous touch events that were not new
     * touchdowns or touchups. */
	for(touchId=0; touchId < FT_NUM_MT_TCH_ID; ++touchId) {
		if(idProcessed & (1 << touchId)) {
			crestSoftKeys_processTouchData(&ts->ctrl, touchId, ts->ctrl.prv_tch[touchId], true);
		}
	}
	// Pickup any stragglers. idProcessed above handles sending the input_sync
	//while looping, this is for any remaining. This is necessary to make sure
	//the last key event is picked up before exiting.
	if (!g_disableInputEvents) {
		input_sync(ts->ctrl.ts_dev);
	}
	ETML_DEBUG( dev_info(&ts->client->dev, "fifoevents = %i\n", fifoEvents); )
}

/* The etml_xy_worker function reads the XY coordinates and sends them to
 * the input layer.  It is scheduled from the interrupt (or timer).
 */
void etml_xy_worker(struct work_struct *work)
{
    struct etml *ts = container_of(work, struct etml, work);
    if(unlikely(ts->do_reset))
    {
    	etml_reset_panel_via_gpio(ts->platform_data->reset_gpio);
    	ts->do_reset = 0;
    }
    else
    {
    	etml_Read_TouchData_1050(ts);

		if (ts->client->irq != 0)
		{
			msleep(ETML_SCAN_DELAY);
	        enable_irq(ts->client->irq);
		}
    }
    return;
}

/*************************************************************************
 * ISR function. This function is general, initialized in drivers init
 * function
 ************************************************************************/
static irqreturn_t etml_irq(int irq, void *handle)
{
    struct etml *ts = (struct etml *) handle;

    disable_irq_nosync(ts->client->irq);
    /* schedule motion signal handling */
    queue_work(etml_ts_wq, &ts->work);
    return IRQ_HANDLED;
}

/* ************************************************************************
 * Probe and Initialization functions
 * ***********************************************************************/

/* etml_initialize: Driver Initialization. This function takes
 * care of the following tasks:
 * 1. Create and register an input device with input layer
 * 2. Take ETML device out of bootloader mode; go operational
 * 3. Start any timers/Work queues.
 * Note that validation has already been performed on the inputs.
 */
static int etml_initialize(struct i2c_client *client, struct etml *ts)
{
	int retval = 0;
	struct input_dev *input_device;
	int id;

	/* Create the input device and register it. */
	input_device = input_allocate_device();
	if (!input_device)
	{
		retval = -ENOMEM;
		printk(KERN_ERR "%s() - ERROR: Could not allocate input device.\n", __FUNCTION__);
		goto error_free_device;
	}

	ts->ctrl.ts_dev = input_device;

	snprintf(ts->phys, sizeof(ts->phys), "%s/input0", dev_name(&client->dev));
	dev_info(&client->dev, "phys = %s\n", ts->phys);
	input_device->name = ETML_NAME;
	input_device->phys = ts->phys;
	input_device->id.bustype = BUS_I2C;
	input_device->dev.parent = &client->dev;
	/* Copy softkeys to the ctrl structure used by crestSoftKeys. */
	ts->ctrl.softkeys = ts->platform_data->softkeys;

	/* Clear the previous touch structures as if no touch */
	for(id=0; id < CRESTRON_MAX_NUM_TOUCHES; ++id) {
		ts->ctrl.prv_tch[id].pos[0] = 0;
		ts->ctrl.prv_tch[id].pos[1] = 0;
		ts->ctrl.prv_tch[id].state = 0;
	}

	ts->ctrl.maxy = ts->platform_data->maxy;
	ts->ctrl.y_bottom_cap = ts->platform_data->y_bottom_cap;

	/* init the touch structures */
	memset(ts->fwname, 0, ETML_QUERY_STR_SIZE);
	memset(ts->fwver, 0, ETML_QUERY_STR_SIZE);

	ts->filter_count = 0;
	ts->tch_filter_rate = 5;

	do_gettimeofday(&ts->last_rd_time);

	set_bit(EV_ABS,    input_device->evbit);
	set_bit(BTN_TOUCH, input_device->keybit);
	retval = input_mt_init_slots(ts->ctrl.ts_dev, CRESTRON_MAX_NUM_TOUCHES);
	if(retval) {
		printk(KERN_ERR "%s() - ERROR: Failed to initialize slots.\n", __FUNCTION__);
		goto error_free_device;
	}
	input_set_abs_params(input_device, ABS_MT_POSITION_X,  0, ts->platform_data->maxx, 0, 0);
	input_set_abs_params(input_device, ABS_MT_POSITION_Y,  0, ts->platform_data->maxy, 0, 0);

	retval = input_register_device(input_device);
	if (0 != retval)
	{
		dev_err(&ts->client->dev, "%s() - ERROR: Could not register input device.\n", __FUNCTION__);
		goto error_free_device;
	}

	/* These are the virtual buttons (need to let android now which ones you will be using) */
	if(ts->platform_data->has_virt_btn) {
		retval = crestSoftKeys_init(&ts->ctrl);
		if(retval) {
			goto error_free_device;
		}
	}

	/* Prepare our worker structure prior to setting up the timer/ISR */
	INIT_WORK(&ts->work, etml_xy_worker);

	input_sync(input_device);

	init_timer(&ts->timer);
	ts->timer.data = (unsigned long) ts;
	ts->timer.function = etml_timer_reset_controller;
	ts->reset_enable = 0;
	ts->reset_time = 24 * 60 * 60; // 1 day in seconds

	/* Interrupt setup */
	if (ts->client->irq)
	{
		/* request_irq() will call enable_irq() */
		retval = request_irq(ts->client->irq, etml_irq, IRQF_TRIGGER_FALLING, input_device->name, ts);
		if (retval)
		{
			printk(KERN_ERR "%s() - ERROR: Could not request IRQ: %d\n", __FUNCTION__, retval);
			goto error_free_irq;
		}
	}

	retval = sysfs_create_group(&client->dev.kobj, &etml_attribute_group);
	if (retval)
	{
		printk(KERN_ERR "%s() - ERROR: sysfs_create_group() failed: %d\n", __FUNCTION__, retval);
	}
	else
	{
		printk(KERN_INFO "%s() - sysfs_create_group() succeeded.\n", __FUNCTION__);
	}

	retval = sysfs_create_group(&input_device->dev.kobj, &etml_touch_attribute_group);
	if (retval)
	{
		printk(KERN_ERR "%s() - ERROR: sysfs_create_group() failed: %d\n", __FUNCTION__, retval);
	}
	else
	{
		printk(KERN_INFO "%s() - sysfs_create_group() succeeded.\n", __FUNCTION__);
	}

	retval = uiTouchIntfInit(&ts->ctrl);
	if(retval) {
		printk(KERN_ERR "%s() - ERROR: Failed it initialize uiTouchIntf.\n", __FUNCTION__);
		goto error_free_irq;
	}
	goto success;

error_free_irq:
	free_irq(ts->client->irq, ts);

error_free_device:
	if (input_device)
	{
		input_free_device(input_device);
		destroy_workqueue(etml_ts_wq);
		ts->ctrl.ts_dev = NULL;
	}

success:
	return retval;
}

/* Function to manage power-on resume */
static int etml_resume(struct i2c_client *client)
{
    int retval = 0;

    printk(KERN_ERR "%s() - Driver is resuming.\n", __FUNCTION__);
#if 0 //RJK not supported yet.
    struct etml *ts = NULL;

    ts = (struct etml *) i2c_get_clientdata(client);
    if(ts->vtp){
    	retval = regulator_enable(ts->vtp);
    }
    if (retval)
    {
        //printk(KERN_ERR "%s() - Could not enable regulator.\n", __FUNCTION__);
    }
    else
    {
		if (ts->platform_data->platform_resume) {
			ts->platform_data->platform_resume();
		}
		etml_reset_panel_via_gpio(ts->platform_data->reset_gpio);
		mb();
		enable_irq(ts->client->irq);
    }
#endif
    return retval;
}


/* Function to manage low power suspend */
static int etml_suspend(struct i2c_client *client, pm_message_t message)
{
    printk(KERN_ERR "%s() - Driver is suspending.\n", __FUNCTION__);

#if 0 //RJK not supported yet
    ts = (struct etml *) i2c_get_clientdata(client);

   /* Disable/enable irq call (in irq/worker function) are matched, disable here for suspend */
    disable_irq(ts->client->irq);

   /* Wait for woker finish, even if worker enables irq, the irq still is disabled because of the above call */
    flush_workqueue(etml_ts_wq);

   /* No need to cancel since irq is disabled, there is no pending worker at this time*/

	if (ts->platform_data->platform_suspend) {
		ts->platform_data->platform_suspend();
	}
	// keep focaltech controller in reset after this point
    gpio_direction_output(ts->platform_data->reset_gpio, 0);
    if(ts->vtp){
    	regulator_disable(ts->vtp);
    }else{
        //printk(KERN_ERR "%s() - Could not disable regulator.\n", __FUNCTION__);
    }
#endif //RJK

    return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void etml_early_suspend(struct early_suspend *handler)
{
    struct etml *ts;
    ts = container_of(handler, struct etml, early_suspend);
    etml_suspend(ts->client, PMSG_SUSPEND);
}

static void etml_late_resume(struct early_suspend *handler)
{
    struct etml *ts;
    ts = container_of(handler, struct etml, early_suspend);
    etml_resume(ts->client);
}
#else  /* CONFIG_HAS_EARLYSUSPEND */
static void etml_early_suspend(struct early_suspend *handler)
{
    /* Do Nothing */
}

static void etml_late_resume(struct early_suspend *handler)
{
    /* Do Nothing */
}
#endif /* CONFIG_HAS_EARLYSUSPEND */

/* I2C driver probe function */
static int __devinit etml_ts_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
    struct etml *ts;
    int retval = 0;
    u8  lbuffer = 0;

    ETML_DEBUG( dev_info(&client->dev, "DEBUG enabled\n"); )

    ts = kzalloc(sizeof(struct etml), GFP_KERNEL);
    if (NULL == ts)
    {
        printk(KERN_ERR "%s() - ERROR: Could not allocate %d bytes of kernel memory for etml struct.\n", __FUNCTION__, sizeof(struct etml));
        retval = -ENOMEM;
        goto error_devinit0;
    }
    msleep(100);

    ts->client = client;
    ts->platform_data = client->dev.platform_data;
    i2c_set_clientdata(client, ts);

    retval = i2c_smbus_read_i2c_block_data(ts->client, 0, sizeof(u8), &lbuffer);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: ETML not found on I2C bus.\n", __FUNCTION__);
        goto error_free_ts;
    }

    printk(KERN_INFO "%s() - ETML found on I2C bus.\n", __FUNCTION__);

    etml_reset_panel_via_gpio(ts->platform_data->reset_gpio);

    // Need to initialize the SYSFS mutex before creating the SYSFS entries in etml_initialize().
    /* TODO: Fix to use the proper mutex. */
    mutex_init(&ts->device_mode_mutex);
    keyStateMutex = kmalloc(sizeof(struct mutex), GFP_KERNEL);
    mutex_init(keyStateMutex);
    //mutex_init(&ts->softkeys.keyStateMutex);

    retval = etml_initialize(client, ts);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Controller could not be initialized.\n", __FUNCTION__);
        goto error_mutex_destroy;
    }

#ifdef CONFIG_HAS_EARLYSUSPEND
    ts->early_suspend.level   = EARLY_SUSPEND_LEVEL_BLANK_SCREEN + 1;
    ts->early_suspend.suspend = etml_early_suspend;
    ts->early_suspend.resume  = etml_late_resume;

    register_early_suspend(&ts->early_suspend);
#endif /* CONFIG_HAS_EARLYSUSPEND */

    goto error_return;

error_devinit0:
error_mutex_destroy:
	kfree(keyStateMutex);
    mutex_destroy(&ts->device_mode_mutex);

error_free_ts:
    kfree(ts);

error_return:
    return retval;
}

/* registered in driver struct */
static int __devexit etml_ts_remove(struct i2c_client *client)
{
    struct etml *ts;

	printk(KERN_INFO "%s() - Driver is unregistering.\n", __FUNCTION__);

    /* clientdata registered on probe */
    ts = i2c_get_clientdata(client);

    /* Start cleaning up by removing any delayed work and the timer */
    if (cancel_delayed_work_sync((struct delayed_work *)&ts->work) < 0)
    {
        printk(KERN_ERR "%s() - ERROR: Could not remove all work from the Work Queue.\n", __FUNCTION__);
    }

    /* free up timer or irq */
    if (ts->client->irq == 0)
    {
    }
    else
    {
        free_irq(client->irq, ts);
    }

#ifdef CONFIG_HAS_EARLYSUSPEND
    unregister_early_suspend(&ts->early_suspend);
#endif /* CONFIG_HAS_EARLYSUSPEND */

    /* housekeeping */
    /* Wait until any outstanding SYSFS transaction has finished,
     * and prevent any new ones from starting.
     */
    mutex_lock(&ts->device_mode_mutex);
    /* Remove the SYSFS entries */
    sysfs_remove_group(&client->dev.kobj, &etml_attribute_group);
    mutex_unlock(&ts->device_mode_mutex);
    mutex_destroy(&ts->device_mode_mutex);

    if (NULL != ts)
    {
        kfree(ts);
    }

    printk(KERN_INFO "%s() - Driver unregistration is complete.\n", __FUNCTION__);

    return 0;
}

static struct i2c_driver etml_ts_driver = {
    .driver = {
        .name = ETML_NAME,
        .owner = THIS_MODULE,
    },
    .probe = etml_ts_probe,
    .remove = __devexit_p(etml_ts_remove),
    .suspend = NULL,
    .resume = NULL,
    .id_table = etml_ts_id,
};

static int etml_ts_init(void)
{
    int ret = 0;

    printk(KERN_INFO "%s() - ETML I2C Touchscreen Driver (Built %s @ %s)\n", __FUNCTION__, __DATE__, __TIME__);

    etml_ts_wq = create_singlethread_workqueue("etml_ts_wq");
    if (NULL == etml_ts_wq)
    {
        printk(KERN_ERR "%s() - ERROR: Could not create the Work Queue due to insufficient memory.\n", __FUNCTION__);
        ret = -ENOMEM;
    }
    else
    {
        ret = i2c_add_driver(&etml_ts_driver);
    }

    return ret;
}


static void etml_ts_exit(void)
{
    if (etml_ts_wq)
    {
        destroy_workqueue(etml_ts_wq);
    }
	i2c_del_driver(&etml_ts_driver);
}

module_init(etml_ts_init);
module_exit(etml_ts_exit);
